% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/slsqp.R
\name{slsqp}
\alias{slsqp}
\title{Sequential Quadratic Programming (SQP)}
\usage{
slsqp(
  x0,
  fn,
  gr = NULL,
  lower = NULL,
  upper = NULL,
  hin = NULL,
  hinjac = NULL,
  heq = NULL,
  heqjac = NULL,
  nl.info = FALSE,
  control = list(),
  deprecatedBehavior = TRUE,
  ...
)
}
\arguments{
\item{x0}{starting point for searching the optimum.}

\item{fn}{objective function that is to be minimized.}

\item{gr}{gradient of function \code{fn}; will be calculated numerically if
not specified.}

\item{lower, upper}{lower and upper bound constraints.}

\item{hin}{function defining the inequality constraints, that is
\code{hin <= 0} for all components. This is new behavior in line with the
rest of the \code{nloptr} arguments. To use the old behavior, please set
\code{deprecatedBehavior} to \code{TRUE}.}

\item{hinjac}{Jacobian of function \code{hin}; will be calculated
numerically if not specified.}

\item{heq}{function defining the equality constraints, that is \code{heq = 0}
for all components.}

\item{heqjac}{Jacobian of function \code{heq}; will be calculated
numerically if not specified.}

\item{nl.info}{logical; shall the original NLopt info been shown.}

\item{control}{list of options, see \code{nl.opts} for help.}

\item{deprecatedBehavior}{logical; if \code{TRUE} (default for now), the old
behavior of the Jacobian function is used, where the equality is \eqn{\ge 0}
instead of \eqn{\le 0}. This will be reversed in a future release and
eventually removed.}

\item{...}{additional arguments passed to the function.}
}
\value{
List with components:
\item{par}{the optimal solution found so far.}
\item{value}{the function value corresponding to \code{par}.}
\item{iter}{number of (outer) iterations, see \code{maxeval}.}
\item{convergence}{integer code indicating successful completion (> 1)
or a possible error number (< 0).}
\item{message}{character string produced by NLopt and giving additional
information.}
}
\description{
Sequential (least-squares) quadratic programming (SQP) algorithm for
nonlinearly constrained, gradient-based optimization, supporting both
equality and inequality constraints.
}
\details{
The algorithm optimizes successive second-order (quadratic/least-squares)
approximations of the objective function (via BFGS updates), with
first-order (affine) approximations of the constraints.
}
\note{
See more infos at
\url{https://nlopt.readthedocs.io/en/latest/NLopt_Algorithms/}.
}
\examples{

##  Solve the Hock-Schittkowski problem no. 100 with analytic gradients
##  See https://apmonitor.com/wiki/uploads/Apps/hs100.apm

x0.hs100 <- c(1, 2, 0, 4, 0, 1, 1)
fn.hs100 <- function(x) {(x[1] - 10) ^ 2 + 5 * (x[2] - 12) ^ 2 + x[3] ^ 4 +
                         3 * (x[4] - 11) ^ 2 + 10 * x[5] ^ 6 + 7 * x[6] ^ 2 +
                         x[7] ^ 4 - 4 * x[6] * x[7] - 10 * x[6] - 8 * x[7]}

hin.hs100 <- function(x) {c(
2 * x[1] ^ 2 + 3 * x[2] ^ 4 + x[3] + 4 * x[4] ^ 2 + 5 * x[5] - 127,
7 * x[1] + 3 * x[2] + 10 * x[3] ^ 2 + x[4] - x[5] - 282,
23 * x[1] + x[2] ^ 2 + 6 * x[6] ^ 2 - 8 * x[7] - 196,
4 * x[1] ^ 2 + x[2] ^ 2 - 3 * x[1] * x[2] + 2 * x[3] ^ 2 + 5 * x[6] -
 11 * x[7])
}

S <- slsqp(x0.hs100, fn = fn.hs100,   # no gradients and jacobians provided
     hin = hin.hs100,
     nl.info = TRUE,
     control = list(xtol_rel = 1e-8, check_derivatives = TRUE),
     deprecatedBehavior = FALSE)

##  The optimum value of the objective function should be 680.6300573
##  A suitable parameter vector is roughly
##  (2.330, 1.9514, -0.4775, 4.3657, -0.6245, 1.0381, 1.5942)

S

}
\references{
Dieter Kraft, ``A software package for sequential quadratic
programming'', Technical Report DFVLR-FB 88-28, Institut fuer Dynamik der
Flugsysteme, Oberpfaffenhofen, July 1988.
}
\seealso{
\code{alabama::auglag}, \code{Rsolnp::solnp},
\code{Rdonlp2::donlp2}
}
\author{
Hans W. Borchers
}
